const express = require('express');
const cors = require('cors');
const path = require('path');
const { Server } = require('socket.io');
const http = require('http');
const crypto = require('crypto');
const session = require('express-session');
const axios = require('axios');
const { ChannelType, PermissionsBitField } = require('discord.js');

/**
 * Creates the Express server with Discord bot integration
 * @param {Client} client - Discord.js client
 * @param {string} guildId - Target guild ID
 * @param {Object} options - Server options
 * @returns {http.Server} HTTP server instance
 */
function createServer(client, guildId, options = {}) {
    const app = express();
    const server = http.createServer(app);

    // Configuration with defaults
    const config = {
        clientId: process.env.DISCORD_CLIENT_ID,
        clientSecret: process.env.DISCORD_CLIENT_SECRET,
        redirectUri: process.env.DISCORD_REDIRECT_URI,
        sessionSecret: process.env.SESSION_SECRET || 'dev-secret-key',
        allowedOrigins: options.allowedOrigins || process.env.ALLOWED_ORIGINS || 'http://localhost:3001',
        enableAuth: options.enableAuth ?? (process.env.ENABLE_AUTH === 'true'),
        batchSize: options.batchSize || 5,
    };

    // Parse allowed origins
    const allowedOriginsList = config.allowedOrigins.split(',').map(o => o.trim());

    // CORS configuration - Security improvement #1
    const corsOptions = {
        origin: (origin, callback) => {
            // Allow requests with no origin (like mobile apps or curl)
            if (!origin) return callback(null, true);

            if (allowedOriginsList.includes(origin) || allowedOriginsList.includes('*')) {
                callback(null, true);
            } else {
                callback(new Error('Not allowed by CORS'));
            }
        },
        methods: ['GET', 'POST'],
        credentials: true,
    };

    const io = new Server(server, { cors: corsOptions });

    // Middleware
    app.use(cors(corsOptions));
    app.use(express.json());
    app.use(express.static(path.join(__dirname, '../public')));

    // Session middleware
    app.use(session({
        secret: config.sessionSecret,
        resave: false,
        saveUninitialized: false,
        cookie: {
            secure: process.env.NODE_ENV === 'production',
            maxAge: 1000 * 60 * 60 * 24 // 24 hours
        }
    }));

    // --- Discord OAuth2 Routes ---

    app.get('/auth/login', (req, res) => {
        if (!config.clientId) {
            return res.status(500).send('OAuth2 is not configured.');
        }
        const state = crypto.randomBytes(16).toString('hex');
        req.session.oauthState = state;
        const url = `https://discord.com/api/oauth2/authorize?client_id=${config.clientId}&redirect_uri=${encodeURIComponent(config.redirectUri)}&response_type=code&scope=identify%20guilds&state=${state}`;
        res.redirect(url);
    });

    app.get('/auth/callback', async (req, res) => {
        const { code, state } = req.query;

        if (state !== req.session.oauthState) {
            return res.status(403).send('State mismatch. Potential CSRF attack.');
        }

        try {
            // Exchange code for token
            const tokenResponse = await axios.post('https://discord.com/api/oauth2/token', new URLSearchParams({
                client_id: config.clientId,
                client_secret: config.clientSecret,
                grant_type: 'authorization_code',
                code,
                redirect_uri: config.redirectUri,
            }), {
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
            });

            const { access_token } = tokenResponse.data;

            // Get user info
            const userResponse = await axios.get('https://discord.com/api/users/@me', {
                headers: { Authorization: `Bearer ${access_token}` }
            });

            req.session.user = userResponse.data;
            req.session.accessToken = access_token;

            res.redirect('/');
        } catch (error) {
            console.error('OAuth2 Error:', error.response?.data || error.message);
            res.status(500).send('Authentication failed.');
        }
    });

    app.get('/auth/logout', (req, res) => {
        req.session.destroy();
        res.redirect('/');
    });

    // API: Get current user
    app.get('/api/me', (req, res) => {
        if (!req.session.user) {
            return res.status(401).json({ success: false, error: 'Not authenticated' });
        }
        res.json({ success: true, user: req.session.user });
    });

    // Authenticated Session Middleware
    const authenticate = (req, res, next) => {
        if (!config.enableAuth) return next();
        if (req.session.user) return next();
        res.status(401).json({ success: false, error: 'Unauthorized' });
    };

    app.use('/api', authenticate);

    // Request logging middleware
    app.use('/api', (req, res, next) => {
        const timestamp = new Date().toISOString();
        console.log(`[${timestamp}] ${req.method} ${req.path}`);
        next();
    });

    // Get the guild with dynamic fallback logic
    const getGuild = (requestedId) => {
        const targetId = requestedId || guildId || process.env.GUILD_ID;
        let guild;

        if (targetId) {
            guild = client.guilds.cache.get(targetId);
        }

        // Fallback to first available if requested/env guild not found
        if (!guild) {
            guild = client.guilds.cache.first();
        }

        if (!guild) {
            throw new Error('Bot is not in any guild or the specified guild ID is invalid.');
        }
        return guild;
    };

    // API: List guilds where user is Admin AND bot is present
    app.get('/api/guilds', async (req, res) => {
        if (!req.session.accessToken) {
            return res.status(401).json({ success: false, error: 'Not authenticated' });
        }

        try {
            // Get user's guilds from Discord API
            const userGuildsRes = await axios.get('https://discord.com/api/users/@me/guilds', {
                headers: { Authorization: `Bearer ${req.session.accessToken}` }
            });

            const userGuilds = userGuildsRes.data;

            // Filter: User has Administrator permission AND Bot is in the guild
            // Administrator bit is 0x8
            const ADMIN_PERMISSION = BigInt(0x8);

            const filteredGuilds = userGuilds
                .filter(ug => {
                    const permissions = BigInt(ug.permissions);
                    const isAdmin = (permissions & ADMIN_PERMISSION) === ADMIN_PERMISSION;
                    const isBotPresent = client.guilds.cache.has(ug.id);
                    return isAdmin && isBotPresent;
                })
                .map(ug => {
                    const botGuild = client.guilds.cache.get(ug.id);
                    return {
                        id: ug.id,
                        name: ug.name,
                        icon: ug.icon ? `https://cdn.discordapp.com/icons/${ug.id}/${ug.icon}.png` : 'https://cdn.discordapp.com/embed/avatars/0.png'
                    };
                });

            res.json({ success: true, guilds: filteredGuilds });
        } catch (error) {
            console.error('Failed to fetch user guilds:', error.response?.data || error.message);
            res.status(500).json({ success: false, error: 'Failed to fetch authorized guilds' });
        }
    });

    // Helper: Process members in parallel batches - Performance improvement
    const processMembersInBatches = async (memberIds, guild, operation) => {
        const results = [];

        for (let i = 0; i < memberIds.length; i += config.batchSize) {
            const batch = memberIds.slice(i, i + config.batchSize);
            const batchResults = await Promise.allSettled(
                batch.map(async (memberId) => {
                    try {
                        const member = await guild.members.fetch(memberId);
                        if (!member.voice.channel) {
                            return { memberId, success: false, error: 'Member is not in a voice channel' };
                        }
                        await operation(member);
                        return { memberId, success: true };
                    } catch (err) {
                        return { memberId, success: false, error: err.message };
                    }
                })
            );

            results.push(...batchResults.map(r =>
                r.status === 'fulfilled' ? r.value : { memberId: 'unknown', success: false, error: r.reason?.message || 'Unknown error' }
            ));
        }

        return results;
    };

    // API: Get all voice channels
    app.get('/api/channels', (req, res) => {
        try {
            const guild = getGuild(req.query.guildId);
            const botMember = guild.members.me;
            const voiceChannels = guild.channels.cache
                .filter((channel) =>
                    channel.type === ChannelType.GuildVoice ||
                    channel.type === ChannelType.GuildStageVoice // Stage channel support
                )
                .map((channel) => ({
                    id: channel.id,
                    name: channel.name,
                    type: channel.type === ChannelType.GuildStageVoice ? 'stage' : 'voice',
                    position: channel.position,
                    memberCount: channel.members.size,
                    // Check bot permissions for this specific channel
                    botCanView: channel.viewable,
                    botCanConnect: channel.connectable,
                    botCanMoveMembers: channel.permissionsFor(botMember).has('MoveMembers'),
                    botCanMuteMembers: channel.permissionsFor(botMember).has('MuteMembers'),
                }))
                .sort((a, b) => a.position - b.position);

            res.json({ success: true, channels: voiceChannels });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Get all members in voice channels
    app.get('/api/members', (req, res) => {
        try {
            const guild = getGuild(req.query.guildId);
            const botMember = guild.members.me;
            const voiceChannels = guild.channels.cache.filter((channel) =>
                channel.type === ChannelType.GuildVoice ||
                channel.type === ChannelType.GuildStageVoice
            );

            const channelsWithMembers = voiceChannels.map((channel) => ({
                channelId: channel.id,
                channelName: channel.name,
                channelType: channel.type === ChannelType.GuildStageVoice ? 'stage' : 'voice',
                position: channel.position,
                permissions: {
                    botCanMove: channel.permissionsFor(botMember).has('MoveMembers'),
                    botCanMute: channel.permissionsFor(botMember).has('MuteMembers'),
                },
                members: channel.members.map((member) => ({
                    id: member.id,
                    username: member.user.username,
                    displayName: member.displayName,
                    avatar: member.user.displayAvatarURL({ size: 64 }),
                    isMuted: member.voice.mute,
                    isDeafened: member.voice.deaf,
                    isSelfMuted: member.voice.selfMute,
                    isSelfDeafened: member.voice.selfDeaf,
                    isSpeaker: member.voice.suppress === false, // For stage channels
                    // Role position constraint: Bot can only move/mute members if its highest role is above theirs
                    botHigherRole: botMember.roles.highest.position > member.roles.highest.position,
                })),
            }))
                .filter((ch) => ch.members.length > 0)
                .sort((a, b) => a.position - b.position);

            res.json({ success: true, channels: channelsWithMembers });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Get all voice channels (including empty) for move target
    app.get('/api/all-channels', (req, res) => {
        try {
            const guild = getGuild(req.query.guildId);
            const voiceChannels = guild.channels.cache
                .filter((channel) =>
                    channel.type === ChannelType.GuildVoice ||
                    channel.type === ChannelType.GuildStageVoice
                )
                .map((channel) => ({
                    id: channel.id,
                    name: channel.name,
                    type: channel.type === ChannelType.GuildStageVoice ? 'stage' : 'voice',
                    position: channel.position,
                }))
                .sort((a, b) => a.position - b.position);

            res.json({ success: true, channels: voiceChannels });
        } catch (error) {
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Move members to a channel - Improved with parallel processing
    app.post('/api/move', async (req, res) => {
        try {
            const { memberIds, targetChannelId, guildId: requestedId } = req.body;

            if (!memberIds || !Array.isArray(memberIds) || memberIds.length === 0) {
                return res.status(400).json({ success: false, error: 'memberIds is required and must be a non-empty array' });
            }

            if (!targetChannelId) {
                return res.status(400).json({ success: false, error: 'targetChannelId is required' });
            }

            const guild = getGuild(requestedId);
            const targetChannel = guild.channels.cache.get(targetChannelId);

            if (!targetChannel ||
                (targetChannel.type !== ChannelType.GuildVoice &&
                    targetChannel.type !== ChannelType.GuildStageVoice)) {
                return res.status(400).json({ success: false, error: 'Invalid target voice channel' });
            }

            console.log(`[MOVE] Moving ${memberIds.length} members to ${targetChannel.name}`);

            const results = await processMembersInBatches(memberIds, guild, async (member) => {
                await member.voice.setChannel(targetChannel);
            });

            const successCount = results.filter(r => r.success).length;
            const failCount = results.filter(r => !r.success).length;
            console.log(`[MOVE] Completed: ${successCount} success, ${failCount} failed`);

            res.json({ success: true, results });
        } catch (error) {
            console.error('[MOVE] Error:', error);
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Mute members - Improved with parallel processing
    app.post('/api/mute', async (req, res) => {
        try {
            const { memberIds, guildId: requestedId } = req.body;

            if (!memberIds || !Array.isArray(memberIds) || memberIds.length === 0) {
                return res.status(400).json({ success: false, error: 'memberIds is required and must be a non-empty array' });
            }

            const guild = getGuild(requestedId);
            console.log(`[MUTE] Muting ${memberIds.length} members`);

            const results = await processMembersInBatches(memberIds, guild, async (member) => {
                await member.voice.setMute(true);
            });

            const successCount = results.filter(r => r.success).length;
            console.log(`[MUTE] Completed: ${successCount} success`);

            res.json({ success: true, results });
        } catch (error) {
            console.error('[MUTE] Error:', error);
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Unmute members - Improved with parallel processing
    app.post('/api/unmute', async (req, res) => {
        try {
            const { memberIds, guildId: requestedId } = req.body;

            if (!memberIds || !Array.isArray(memberIds) || memberIds.length === 0) {
                return res.status(400).json({ success: false, error: 'memberIds is required and must be a non-empty array' });
            }

            const guild = getGuild(requestedId);
            console.log(`[UNMUTE] Unmuting ${memberIds.length} members`);

            const results = await processMembersInBatches(memberIds, guild, async (member) => {
                await member.voice.setMute(false);
            });

            const successCount = results.filter(r => r.success).length;
            console.log(`[UNMUTE] Completed: ${successCount} success`);

            res.json({ success: true, results });
        } catch (error) {
            console.error('[UNMUTE] Error:', error);
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Disconnect members from voice - Improved with parallel processing
    app.post('/api/disconnect', async (req, res) => {
        try {
            const { memberIds, guildId: requestedId } = req.body;

            if (!memberIds || !Array.isArray(memberIds) || memberIds.length === 0) {
                return res.status(400).json({ success: false, error: 'memberIds is required and must be a non-empty array' });
            }

            const guild = getGuild(requestedId);
            console.log(`[DISCONNECT] Disconnecting ${memberIds.length} members`);

            const results = await processMembersInBatches(memberIds, guild, async (member) => {
                await member.voice.disconnect();
            });

            const successCount = results.filter(r => r.success).length;
            console.log(`[DISCONNECT] Completed: ${successCount} success`);

            res.json({ success: true, results });
        } catch (error) {
            console.error('[DISCONNECT] Error:', error);
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Batch set regional RTC for all voice channels
    app.post('/api/batch-set-region', async (req, res) => {
        try {
            const { guildId: requestedId, region } = req.body;
            const guild = getGuild(requestedId);
            
            console.log(`[REGION] Setting RTC region for all channels in ${guild.name} to: ${region || 'automatic'}`);
            
            const voiceChannels = guild.channels.cache.filter((channel) =>
                channel.type === ChannelType.GuildVoice ||
                channel.type === ChannelType.GuildStageVoice
            );

            const results = await Promise.allSettled(
                voiceChannels.map(async (channel) => {
                    try {
                        await channel.setRTCRegion(region || null);
                        return { channelId: channel.id, channelName: channel.name, success: true };
                    } catch (err) {
                        return { channelId: channel.id, channelName: channel.name, success: false, error: err.message };
                    }
                })
            );

            const processedResults = results.map(r => 
                r.status === 'fulfilled' ? r.value : { success: false, error: r.reason?.message }
            );

            const successCount = processedResults.filter(r => r.success).length;
            console.log(`[REGION] Completed: ${successCount}/${voiceChannels.size} channels updated`);

            res.json({ success: true, results: processedResults });
        } catch (error) {
            console.error('[REGION] Error:', error);
            res.status(500).json({ success: false, error: error.message });
        }
    });

    // API: Get bot invite URL
    app.get('/api/invite', (req, res) => {
        const clientId = process.env.DISCORD_CLIENT_ID;
        if (!clientId) {
            return res.status(500).json({ success: false, error: 'Client ID not configured' });
        }
        // Permissions: View Channels(1024) + Connect(1048576) + Speak(2097152) + Mute(4194304) + Deafen(8388608) + Move(16777216) = 32508928
        const inviteUrl = `https://discord.com/api/oauth2/authorize?client_id=${clientId}&permissions=32508928&scope=bot%20applications.commands`;
        res.json({ success: true, url: inviteUrl });
    });

    // API: Health check endpoint
    app.get('/api/health', (req, res) => {
        try {
            const guild = getGuild(req.query.guildId);
            res.json({
                success: true,
                status: 'healthy',
                bot: {
                    username: client.user?.tag,
                    ready: client.isReady(),
                },
                guild: {
                    id: guild.id,
                    name: guild.name,
                    memberCount: guild.memberCount,
                },
            });
        } catch (error) {
            res.status(503).json({
                success: false,
                status: 'unhealthy',
                error: error.message,
            });
        }
    });

    // Global error handler - Stability improvement
    app.use((err, req, res, next) => {
        console.error('[ERROR]', err);
        res.status(500).json({
            success: false,
            error: process.env.NODE_ENV === 'production' ? 'Internal server error' : err.message,
        });
    });

    // Socket.io for real-time updates
    io.on('connection', (socket) => {
        console.log('🔌 Client connected:', socket.id);

        // Send initial health status
        try {
            const guild = getGuild();
            socket.emit('healthStatus', {
                connected: true,
                guild: guild.name,
            });
        } catch (error) {
            socket.emit('healthStatus', {
                connected: false,
                error: error.message,
            });
        }

        socket.on('disconnect', () => {
            console.log('🔌 Client disconnected:', socket.id);
        });
    });

    // Debounce voice updates to prevent flooding
    let voiceUpdateTimeout = null;
    const VOICE_UPDATE_DEBOUNCE_MS = 200;

    // Listen for voice state updates and broadcast to clients
    client.on('voiceUpdate', ({ oldState, newState }) => {
        // Debounce rapid updates
        if (voiceUpdateTimeout) {
            clearTimeout(voiceUpdateTimeout);
        }

        voiceUpdateTimeout = setTimeout(() => {
            io.emit('voiceUpdate', {
                userId: newState.member?.id || oldState.member?.id,
                oldChannelId: oldState.channelId,
                newChannelId: newState.channelId,
                timestamp: Date.now(),
            });
        }, VOICE_UPDATE_DEBOUNCE_MS);
    });

    return server;
}

module.exports = createServer;
