// Discord VC Manager - Frontend JavaScript

class VCManager {
    constructor() {
        this.selectedMembers = new Set();
        this.channels = [];
        this.allChannels = [];
        this.guilds = [];
        this.currentGuildId = null;
        this.isLoading = false;
        this.lastLoadTime = 0;
        this.loadDebounceMs = 300;
        this.searchQuery = '';
        this.user = null;

        // DOM Elements
        this.channelsContainer = document.getElementById('channelsContainer');
        this.selectedCount = document.getElementById('selectedCount');
        this.targetChannel = document.getElementById('targetChannel');
        this.moveBtn = document.getElementById('moveBtn');
        this.muteBtn = document.getElementById('muteBtn');
        this.unmuteBtn = document.getElementById('unmuteBtn');
        this.disconnectBtn = document.getElementById('disconnectBtn');
        this.selectAllBtn = document.getElementById('selectAllBtn');
        this.deselectAllBtn = document.getElementById('deselectAllBtn');
        this.regionSelector = document.getElementById('regionSelector');
        this.batchRegionBtn = document.getElementById('batchRegionBtn');
        this.connectionStatus = document.getElementById('connectionStatus');
        this.toastContainer = document.getElementById('toastContainer');
        this.memberSearch = document.getElementById('memberSearch');
        this.guildSelector = document.getElementById('guildSelector');
        this.authOverlay = document.getElementById('authOverlay');
        this.userProfile = document.getElementById('userProfile');
        this.userAvatar = document.getElementById('userAvatar');
        this.userName = document.getElementById('userName');
        this.inviteLink = document.getElementById('inviteLink');

        // Initialize
        this.init();
    }

    async init() {
        const authenticated = await this.checkAuth();
        if (!authenticated) return;

        this.loadSelection();
        this.loadGuildFromStorage();
        this.setupEventListeners();
        this.setupSocketIO();
        this.setupDragAndDrop();
        this.loadInviteUrl(); // Start loading URL early
        await this.loadInitialData();
    }

    async checkAuth() {
        try {
            const res = await fetch('/api/me');
            const data = await res.json();
            if (data.success) {
                this.user = data.user;
                this.renderUserProfile();
                this.authOverlay.classList.add('hidden');
                return true;
            } else {
                this.showLogin();
                return false;
            }
        } catch (e) {
            console.error('Auth check failed', e);
            this.showLogin();
            return false;
        }
    }

    showLogin() {
        this.authOverlay.classList.remove('hidden');
        this.userProfile.classList.add('hidden');
    }

    renderUserProfile() {
        if (!this.user) return;
        this.userAvatar.src = this.user.avatar ?
            `https://cdn.discordapp.com/avatars/${this.user.id}/${this.user.avatar}.png` :
            'https://cdn.discordapp.com/embed/avatars/0.png';
        this.userName.textContent = this.user.global_name || this.user.username;
        this.userProfile.classList.remove('hidden');
    }

    async loadInitialData() {
        await this.loadGuilds();
        if (!this.currentGuildId && this.guilds.length > 0) {
            this.currentGuildId = this.guilds[0].id;
            this.guildSelector.value = this.currentGuildId;
        }
        await this.loadData();
    }

    async loadInviteUrl() {
        console.log('Fetching invite URL...');
        try {
            const res = await fetch('/api/invite');
            const data = await res.json();
            if (data.success) {
                console.log('Invite URL loaded:', data.url);
                this.inviteLink.href = data.url;
            } else {
                console.error('Failed to get invite URL:', data.error);
                this.showToast('招待リンクの取得に失敗しました: ' + data.error, 'error');
            }
        } catch (e) {
            console.error('Error fetching invite URL:', e);
            this.showToast('招待リンク取得中にエラーが発生しました', 'error');
        }
    }

    setupEventListeners() {
        this.moveBtn.addEventListener('click', () => this.handleAction('move'));
        this.muteBtn.addEventListener('click', () => this.handleAction('mute'));
        this.unmuteBtn.addEventListener('click', () => this.handleAction('unmute'));
        this.disconnectBtn.addEventListener('click', () => this.handleAction('disconnect'));
        this.selectAllBtn.addEventListener('click', () => this.selectAll());
        this.deselectAllBtn.addEventListener('click', () => this.deselectAll());
        this.targetChannel.addEventListener('change', () => this.updateButtons());

        this.memberSearch.addEventListener('input', (e) => {
            this.searchQuery = e.target.value.toLowerCase();
            this.renderChannels();
        });

        this.inviteLink.addEventListener('click', (e) => {
            const href = this.inviteLink.getAttribute('href');
            if (!href || href === '#' || !href.startsWith('https://discord.com')) {
                e.preventDefault();
                console.log('Invite link not ready yet or invalid:', href);
                this.showToast('招待リンクを準備中です。', 'warning');
            }
        });

        this.guildSelector.addEventListener('change', (e) => {
            this.currentGuildId = e.target.value;
            this.saveGuildToStorage();
            this.deselectAll(); // Clear selection when switching guilds
            this.loadData();
        });

        this.batchRegionBtn.addEventListener('click', () => this.handleBatchRegion());
    }

    setupSocketIO() {
        this.socket = io();

        this.socket.on('connect', () => {
            this.updateConnectionStatus('connected');
        });

        this.socket.on('disconnect', () => {
            this.updateConnectionStatus('disconnected');
        });

        this.socket.on('healthStatus', (data) => {
            if (data.connected) {
                console.log('Bot connection healthy');
            } else {
                this.showToast('Botの接続に問題があります: ' + data.error, 'error');
            }
        });

        this.socket.on('voiceUpdate', (data) => {
            // Potentially filter voice update by currentGuildId if the server sends it
            this.debouncedLoadMembers();
        });
    }

    setupDragAndDrop() {
        this.channelsContainer.addEventListener('dragover', (e) => {
            const card = e.target.closest('.channel-card');
            if (card) {
                e.preventDefault();
                card.classList.add('drag-over');
            }
        });

        this.channelsContainer.addEventListener('dragleave', (e) => {
            const card = e.target.closest('.channel-card');
            if (card) {
                card.classList.remove('drag-over');
            }
        });

        this.channelsContainer.addEventListener('drop', async (e) => {
            const card = e.target.closest('.channel-card');
            if (card) {
                e.preventDefault();
                card.classList.remove('drag-over');
                const memberId = e.dataTransfer.getData('text/plain');
                const targetChannelId = card.dataset.channelId;

                if (memberId && targetChannelId) {
                    let membersToMove = [memberId];
                    if (this.selectedMembers.has(memberId)) {
                        membersToMove = Array.from(this.selectedMembers);
                    }

                    await this.apiCall('/api/move', {
                        memberIds: membersToMove,
                        targetChannelId,
                        guildId: this.currentGuildId
                    }, `移動中...`);

                    if (this.selectedMembers.has(memberId)) {
                        this.deselectAll();
                    }
                    await this.loadMembers();
                }
            }
        });
    }

    debouncedLoadMembers() {
        const now = Date.now();
        if (now - this.lastLoadTime < this.loadDebounceMs) return;
        this.lastLoadTime = now;
        this.loadMembers();
    }

    updateConnectionStatus(status) {
        this.connectionStatus.className = `connection-status ${status}`;
        const statusText = this.connectionStatus.querySelector('.status-text');
        statusText.textContent = status === 'connected' ? '接続済み' : (status === 'disconnected' ? '切断' : '接続中...');
    }

    async loadData() {
        if (!this.currentGuildId) return;
        await Promise.all([
            this.loadMembers(),
            this.loadAllChannels()
        ]);
    }

    async loadGuilds() {
        try {
            const res = await fetch('/api/guilds');
            if (res.status === 401) {
                this.showLogin();
                return;
            }
            const data = await res.json();
            if (data.success) {
                this.guilds = data.guilds;
                this.renderGuildSelector();
                if (!this.currentGuildId && this.guilds.length > 0) {
                    this.currentGuildId = this.guilds[0].id;
                    this.guildSelector.value = this.currentGuildId;
                    this.loadData();
                }
            }
        } catch (e) {
            console.error('Failed to load guilds', e);
        }
    }

    renderGuildSelector() {
        this.guildSelector.innerHTML = this.guilds.map(guild => `
      <option value="${guild.id}" ${this.currentGuildId === guild.id ? 'selected' : ''}>
        ${this.escapeHtml(guild.name)}
      </option>
    `).join('') || '<option value="">サーバーが見つかりません</option>';
    }

    async loadMembers() {
        if (this.isLoading || !this.currentGuildId) return;

        try {
            this.isLoading = true;
            const response = await fetch(`/api/members?guildId=${this.currentGuildId}`);
            if (response.status === 401) {
                this.showLogin();
                return;
            }
            const data = await response.json();

            if (data.success) {
                this.channels = data.channels;
                this.renderChannels();
            } else {
                throw new Error(data.error);
            }
        } catch (error) {
            console.error('Failed to load members:', error);
            this.channelsContainer.innerHTML = `<div class="empty-state"><p>読み込みエラー: ${error.message}</p></div>`;
        } finally {
            this.isLoading = false;
        }
    }

    async loadAllChannels() {
        if (!this.currentGuildId) return;
        try {
            const response = await fetch(`/api/all-channels?guildId=${this.currentGuildId}`);
            const data = await response.json();
            if (data.success) {
                this.allChannels = data.channels;
                this.renderChannelSelect();
            }
        } catch (error) {
            console.error('Failed to load channels:', error);
        }
    }

    renderChannelSelect() {
        this.targetChannel.innerHTML = '<option value="">移動先を選択...</option>' +
            this.allChannels.map(ch => `
        <option value="${ch.id}">${ch.type === 'stage' ? '📢' : '🔊'} ${this.escapeHtml(ch.name)}</option>
      `).join('');
        this.targetChannel.disabled = false;
    }

    renderChannels() {
        if (this.channels.length === 0) {
            this.channelsContainer.innerHTML = '<div class="empty-state"><p>このサーバーのボイスチャンネルには誰もいません</p></div>';
            return;
        }

        const filteredChannels = this.channels.map(channel => ({
            ...channel,
            members: channel.members.filter(m =>
                m.displayName.toLowerCase().includes(this.searchQuery) ||
                m.username.toLowerCase().includes(this.searchQuery)
            )
        })).filter(channel => channel.members.length > 0 || this.searchQuery === '');

        if (filteredChannels.length === 0 && this.searchQuery !== '') {
            this.channelsContainer.innerHTML = '<div class="empty-state"><p>一致するメンバーが見つかりません</p></div>';
            return;
        }

        this.channelsContainer.innerHTML = filteredChannels.map(channel => {
            const icon = channel.channelType === 'stage' ?
                '<svg class="channel-icon" viewBox="0 0 24 24" fill="currentColor"><path d="M19 8H5c-1.1 0-2 .9-2 2v4c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2v-4c0-1.1-.9-2-2-2zm-7 6c-1.66 0-3-1.34-3-3s1.34-3 3-3 3 1.34 3 3-1.34 3-3 3z"/></svg>' :
                '<svg class="channel-icon" viewBox="0 0 24 24" fill="currentColor"><path d="M12 3v10.55c-.59-.34-1.27-.55-2-.55-2.21 0-4 1.79-4 4s1.79 4 4 4 4-1.79 4-4V7h4V3h-6z"/></svg>';

            const botCanManage = channel.permissions?.botCanMove || channel.permissions?.botCanMute;
            const botPermissionWarning = !botCanManage ?
                '<span class="permission-warning" title="このチャンネルに対する管理権限がBotにありません">⚠️</span>' : '';

            const isAllSelectedInVC = channel.members.length > 0 && channel.members.every(m => this.selectedMembers.has(m.id));

            return `
        <div class="channel-card" data-channel-id="${channel.channelId}">
          <div class="channel-header">
            ${icon}
            <span class="channel-name">${this.escapeHtml(channel.channelName)}</span>
            ${channel.channelType === 'stage' ? '<span class="channel-type-badge">ステージ</span>' : ''}
            ${botPermissionWarning}
            <div class="vc-actions">
              <button class="vc-select-btn ${isAllSelectedInVC ? 'active' : ''}" title="このVCの全員を選択/解除" onclick="event.stopPropagation(); vcManager.toggleVCSelection('${channel.channelId}')">
                <svg viewBox="0 0 24 24" fill="currentColor"><path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41L9 16.17z"/></svg>
              </button>
            </div>
            <span class="member-count">${channel.members.length} 人</span>
          </div>
          <div class="channel-members">
            ${channel.members.map(member => this.renderMember(member)).join('')}
          </div>
        </div>
      `;
        }).join('');

        this.attachMemberListeners();
        this.updateSelectedCount();
    }

    attachMemberListeners() {
        this.channelsContainer.querySelectorAll('.member-item').forEach(item => {
            item.addEventListener('click', (e) => this.toggleMember(e.currentTarget));
            item.addEventListener('dragstart', (e) => {
                e.dataTransfer.setData('text/plain', e.currentTarget.dataset.memberId);
                e.currentTarget.classList.add('dragging');
            });
            item.addEventListener('dragend', (e) => e.currentTarget.classList.remove('dragging'));
        });
    }

    renderMember(member) {
        const isSelected = this.selectedMembers.has(member.id);
        const noPermissionClass = !member.botHigherRole ? 'no-permission' : '';
        const roleWarning = !member.botHigherRole ?
            '<svg class="permission-warning" title="Botのロール順位が低いため操作できません" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>' : '';

        const nameLength = (member.displayName || '').length;
        let nameClass = '';
        if (nameLength > 20) nameClass = 'name-ultra-long';
        else if (nameLength > 15) nameClass = 'name-very-long';
        else if (nameLength > 10) nameClass = 'name-long';

        return `
      <div class="member-item ${isSelected ? 'selected' : ''} ${noPermissionClass}" 
           data-member-id="${member.id}" draggable="true">
        <div class="member-checkbox">
          <svg viewBox="0 0 24 24" fill="currentColor"><path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41L9 16.17z"/></svg>
        </div>
        <img class="member-avatar" src="${member.avatar}" alt="${this.escapeHtml(member.displayName)}" onerror="this.src='https://cdn.discordapp.com/embed/avatars/0.png'">
        <div class="member-info">
          <div class="member-name ${nameClass}" data-length="${nameLength}">${this.escapeHtml(member.displayName)} ${roleWarning}</div>
        </div>
        <div class="member-status">
          ${member.isMuted ? '<svg class="status-icon muted" viewBox="0 0 24 24" fill="currentColor"><path d="M16.5 12c0-1.77-1.02-3.29-2.5-4.03v2.21l2.45 2.45c.03-.2.05-.41.05-.63zm2.5 0c0 .94-.2 1.82-.54 2.64l1.51 1.51C20.63 14.91 21 13.5 21 12c0-4.28-2.99-7.86-7-8.77v2.06c2.89.86 5 3.54 5 6.71zM4.27 3L3 4.27 7.73 9H3v6h4l5 5v-6.73l4.25 4.25c-.67.52-1.42.93-2.25 1.18v2.06c1.38-.31 2.63-.95 3.69-1.81L19.73 21 21 19.73l-9-9L4.27 3zM12 4L9.91 6.09 12 8.18V4z"/></svg>' : ''}
          ${member.isSelfMuted ? '<svg class="status-icon self-muted" viewBox="0 0 24 24" fill="currentColor"><path d="M19 11h-1.7c0 .74-.16 1.43-.43 2.05l1.23 1.23c.56-.98.9-2.09.9-3.28zm-4.02.17c0-.06.02-.11.02-.17V5c0-1.66-1.34-3-3-3S9 3.34 9 5v.18l5.98 5.99zM4.27 3L3 4.27l6.01 6.01V11c0 1.66 1.33 3 2.99 3 .22 0 .44-.03.65-.08l1.66 1.66c-.71.33-1.5.52-2.31.52-2.76 0-5.3-2.1-5.3-5.1H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c.91-.13 1.77-.45 2.54-.9L19.73 21 21 19.73 4.27 3z"/></svg>' : ''}
        </div>
      </div>
    `;
    }

    toggleMember(element) {
        const id = element.dataset.memberId;
        if (this.selectedMembers.has(id)) {
            this.selectedMembers.delete(id);
            element.classList.remove('selected');
        } else {
            this.selectedMembers.add(id);
            element.classList.add('selected');
        }
        this.saveSelection();
        this.updateSelectedCount();
    }

    selectAll() {
        this.channelsContainer.querySelectorAll('.member-item').forEach(item => {
            this.selectedMembers.add(item.dataset.memberId);
            item.classList.add('selected');
        });
        this.saveSelection();
        this.updateSelectedCount();
    }

    deselectAll() {
        this.selectedMembers.clear();
        this.renderChannels(); // More robust than manual class removal
        this.saveSelection();
        this.updateSelectedCount();
    }

    toggleVCSelection(channelId) {
        const channel = this.channels.find(c => c.channelId === channelId);
        if (!channel) return;

        const allIds = channel.members.map(m => m.id);
        const isAllInVCSelected = allIds.every(id => this.selectedMembers.has(id));

        if (isAllInVCSelected) {
            allIds.forEach(id => this.selectedMembers.delete(id));
        } else {
            allIds.forEach(id => this.selectedMembers.add(id));
        }

        this.renderChannels();
        this.saveSelection();
        this.updateSelectedCount();
    }

    saveSelection() {
        localStorage.setItem(`selected_${this.currentGuildId}`, JSON.stringify(Array.from(this.selectedMembers)));
    }

    loadSelection() {
        const saved = localStorage.getItem(`selected_${this.currentGuildId}`);
        if (saved) this.selectedMembers = new Set(JSON.parse(saved));
    }

    saveGuildToStorage() {
        localStorage.setItem('last_guild_id', this.currentGuildId);
    }

    loadGuildFromStorage() {
        this.currentGuildId = localStorage.getItem('last_guild_id');
    }

    updateSelectedCount() {
        this.selectedCount.textContent = this.selectedMembers.size;
        this.updateButtons();
    }

    updateButtons() {
        const hasSelection = this.selectedMembers.size > 0;
        this.moveBtn.disabled = !hasSelection || !this.targetChannel.value;
        this.muteBtn.disabled = !hasSelection;
        this.unmuteBtn.disabled = !hasSelection;
        this.disconnectBtn.disabled = !hasSelection;
    }

    async handleAction(action) {
        const memberIds = Array.from(this.selectedMembers);
        if (memberIds.length === 0) return;
        if (memberIds.length >= 5 && !confirm(`${memberIds.length}名に対して操作を実行しますか？`)) return;

        const payload = { memberIds, guildId: this.currentGuildId };
        if (action === 'move') {
            payload.targetChannelId = this.targetChannel.value;
            await this.apiCall('/api/move', payload, '移動中...');
            this.deselectAll();
        } else if (action === 'disconnect') {
            if (confirm(`${memberIds.length}名をチャンネルから切断しますか？`)) {
                await this.apiCall('/api/disconnect', payload, '切断中...');
                this.deselectAll();
            }
        } else {
            await this.apiCall(`/api/${action}`, payload, action === 'mute' ? 'ミュート中...' : '解除中...');
        }
        await this.loadMembers();
    }

    async handleBatchRegion() {
        const region = this.regionSelector.value;
        const regionLabel = this.regionSelector.options[this.regionSelector.selectedIndex].text;

        if (!confirm(`すべてのボイスチャンネルの地域を「${regionLabel}」に変更しますか？`)) return;

        const payload = {
            guildId: this.currentGuildId,
            region: region === "" ? null : region
        };

        this.setButtonLoading(this.batchRegionBtn, true, '設定中...');
        try {
            const res = await fetch('/api/batch-set-region', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            const data = await res.json();
            if (data.success) {
                const results = data.results || [];
                const successCount = results.filter(r => r.success).length;
                this.showToast(`完了: ${successCount}個のチャンネルを更新しました`, 'success');
            } else throw new Error(data.error);
        } catch (e) {
            this.showToast('エラー: ' + e.message, 'error');
        } finally {
            this.setButtonLoading(this.batchRegionBtn, false);
        }
    }

    async apiCall(url, body, loadingText) {
        let btn;
        if (url.includes('move')) btn = this.moveBtn;
        else if (url.includes('unmute')) btn = this.unmuteBtn;
        else if (url.includes('mute')) btn = this.muteBtn;
        else if (url.includes('disconnect')) btn = this.disconnectBtn;

        this.setButtonLoading(btn, true, loadingText);
        try {
            const res = await fetch(url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(body)
            });
            const data = await res.json();
            if (data.success) {
                this.showToast('完了しました', 'success');
            } else throw new Error(data.error);
        } catch (e) {
            this.showToast('エラー: ' + e.message, 'error');
        } finally {
            this.setButtonLoading(btn, false);
        }
    }

    setButtonLoading(btn, loading, text) {
        if (loading) {
            btn.dataset.prevHtml = btn.innerHTML;
            btn.innerHTML = `<span class="spinner"></span> ${text}`;
            btn.disabled = true;
        } else {
            btn.innerHTML = btn.dataset.prevHtml || btn.innerHTML;
            btn.disabled = false;
        }
    }

    showToast(message, type = 'success') {
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.textContent = message;
        this.toastContainer.appendChild(toast);
        setTimeout(() => toast.remove(), 3000);
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

document.addEventListener('DOMContentLoaded', () => {
    window.vcManager = new VCManager();
});
